<?php

namespace App\Filament\Resources;

use App\Filament\Resources\MenuResource\Pages;
use App\Filament\Resources\MenuResource\RelationManagers;
use App\Models\Menu;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class MenuResource extends Resource
{
    protected static ?string $model = Menu::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'Menü';


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('label')
                    ->label('Menü Adı')
                    ->required(),

                    Forms\Components\Select::make('type')
                        ->label('Bağlantı Türü')
                        ->options([
                            'page' => 'Sayfa (Page)',
                            'news' => 'Haber Listesi (News)',
                            'custom' => 'Özel URL',
                        ])
                        ->default('page')
                        ->required()
                        ->reactive(), // ❗ Bu önemli: değer değişince formu tetikler

                        Forms\Components\Select::make('page_id')
                          ->label('Bağlı Sayfa')
                          ->options(\App\Models\Page::all()->pluck('title', 'id'))
                          ->searchable()
                          ->visible(fn (callable $get) => $get('type') === 'page'), // ❗ Sadece type = page olduğunda göster






                Forms\Components\TextInput::make('url')
                    ->label('Direkt URL (opsiyonel)')
                    ->helperText('Eğer özel URL istiyorsan burayı doldur, yoksa bağlı sayfa açılır.'),

                Forms\Components\Select::make('parent_id')
                    ->label('Üst Menü')
                    ->options(\App\Models\Menu::all()->pluck('label', 'id'))
                    ->searchable()
                    ->nullable(),

                Forms\Components\TextInput::make('sort')
                    ->label('Sıralama')
                    ->numeric()
                    ->default(0),

                      Forms\Components\FileUpload::make('images')
    ->label('Menü Görseli')
    ->image()
    ->directory('menu-images') // storage/app/public/menu-images klasörüne kaydedilir
    ->imageEditor() // (isteğe bağlı) küçük bir kırpma vs. imkanı verir
    ->imagePreviewHeight('150') // ön izleme boyutu
    ->maxSize(1024) // 1MB sınırı (isteğe bağlı)
    ->helperText('Menüye ait görsel yükleyebilirsiniz (isteğe bağlı).')
    ->nullable(),
            ]);
    }


    public static function table(Tables\Table $table): Tables\Table
  {
      return $table
          ->query(self::buildMenuTreeQuery())
          ->columns([
              Tables\Columns\TextColumn::make('indented_label')
                  ->label('Menü')
                  ->sortable(false),  // sıralama bizde zaten

              Tables\Columns\TextColumn::make('parent.label')
                  ->label('Üst Menü'),

              Tables\Columns\TextColumn::make('page.title')
                  ->label('Bağlı Sayfa'),
          ])
          ->defaultSort('sort')
          ->actions([
              Tables\Actions\EditAction::make(),
          ]);
  }

  protected static function buildMenuTreeQuery()
  {
      $menus = \App\Models\Menu::with('children', 'parent', 'page')
                  ->whereNull('parent_id')
                  ->orderBy('sort')
                  ->get();

      $flattened = collect();

      foreach ($menus as $menu) {
          $flattened->push($menu);
          self::addChildrenRecursive($flattened, $menu);
      }

      $ids = $flattened->pluck('id');

      return \App\Models\Menu::whereIn('id', $ids)->orderByRaw('FIELD(id, ' . $ids->implode(',') . ')');
  }

  protected static function addChildrenRecursive(&$flattened, $menu)
  {
      foreach ($menu->children()->orderBy('sort')->get() as $child) {
          $flattened->push($child);
          self::addChildrenRecursive($flattened, $child);
      }
  }


    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMenus::route('/'),
            'create' => Pages\CreateMenu::route('/create'),
            'edit' => Pages\EditMenu::route('/{record}/edit'),
        ];
    }
}
