<?php

namespace App\Filament\Resources;

use App\Filament\Resources\NewsResource\Pages;
use App\Filament\Resources\NewsResource\RelationManagers;
use App\Models\News;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\ToggleColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\Storage;
use Filament\Forms\Components\RichEditor;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver as GdDriver;


class NewsResource extends Resource
{
    protected static ?string $model = News::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'Haberler';


    public static function form(Forms\Form $form): Forms\Form
    {
        return $form
            ->schema([
              Forms\Components\TextInput::make('title')
                  ->label('Başlık')
                  ->required()
                  ->reactive()
                  ->afterStateUpdated(function ($state, callable $set) {
                      $set('slug', \Str::slug($state));
                  }),

              Forms\Components\TextInput::make('slug')
                  ->label('URL Slug')
                  ->required(),

                    Forms\Components\RichEditor::make('content')
                        ->label('İçerik')
                        ->required()
                        ->columnSpanFull(),

                        Forms\Components\FileUpload::make('image')
                            ->label('Görsel')
                            ->directory('pages/news')
                            ->image()
                            ->deletable()
                            ->reorderable(false)
                            ->columnSpanFull()
                            ->afterStateUpdated(function ($state) {
                                if (!$state) return;

                                $fullPath = Storage::disk('public')->path($state);
                                $filename = basename($state);

                                // Dosya mevcut mu?
                                if (!file_exists($fullPath)) {
                                    \Log::error("Resim bulunamadı: " . $fullPath);
                                    return;
                                }

                                try {
                                    $manager = new \Intervention\Image\ImageManager(
                                        new \Intervention\Image\Drivers\Gd\Driver()
                                    );

                                    $image = $manager->read($fullPath);

                                    $image->cover(789, 400)
                                        ->save(Storage::disk('public')->path('news/789x400/' . $filename));

                                    $image->cover(384, 271)
                                        ->save(Storage::disk('public')->path('news/384x271/' . $filename));
                                } catch (\Intervention\Image\Exceptions\DecoderException $e) {
                                    \Log::error("Resim decode edilemedi: " . $e->getMessage());
                                }
                            }),


                Forms\Components\Toggle::make('status')
                    ->label('Yayında mı?')
                    ->default(true),



            ]);
    }
    public static function table(Table $table): Table
    {
        return $table
                ->defaultSort('created_at', 'desc') // Son eklenen en üstte
            ->columns([
                ImageColumn::make('image')
                    ->label('Görsel')
                    ->disk('public')
                    ->height(50)
                    ->width(50),

                TextColumn::make('title')
                    ->label('Başlık')
                        ->limit(40) // Maksimum 40 karakter gösterir
                    ->searchable()
                    ->sortable(),

                TextColumn::make('slug')
                    ->label('Slug')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),

                    TextColumn::make('gallery')
    ->label('Galeri')
    ->formatStateUsing(fn ($state) => is_array(json_decode($state, true)) ? count(json_decode($state, true)) . ' görsel' : '0'),


    TextColumn::make('views')
           ->label('Okunma')
           ->sortable(),


                ToggleColumn::make('status')
                    ->label('Yayında mı?')
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('Oluşturulma')
                    ->dateTime('d.m.Y H:i')
                    ->sortable()
                    ->since(), // örn: "3 gün önce"
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('status')
                    ->label('Yayın Durumu')
                    ->trueLabel('Yayında')
                    ->falseLabel('Taslak'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListNews::route('/'),
            'create' => Pages\CreateNews::route('/create'),
            'edit' => Pages\EditNews::route('/{record}/edit'),
        ];
    }


}
