<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ProjectResource\Pages;
use App\Filament\Resources\ProjectResource\RelationManagers;
use App\Models\Project;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Actions\Action;
use App\Models\News;

class ProjectResource extends Resource
{
    protected static ?string $model = Project::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationLabel = 'Etkinlikler';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make(2)
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label('Proje Başlığı')
                            ->required()
                            ->reactive()
                            ->afterStateUpdated(fn ($state, callable $set) => $set('slug', \Str::slug($state))),

                        Forms\Components\TextInput::make('breadcrumb')
                            ->label('Breadcumb Başlık')
                            ->required(),

                            Forms\Components\TextInput::make('etkinliktarih')
                                ->label('Etkinlik Tarihi')
                                ->required(),

    Forms\Components\TextInput::make('etkinlikkonum')
        ->label('Etkinlik Konumu')
        ->required(),

        Forms\Components\Select::make('related_news_id')
            ->label('Etkinlik Haberi')
            ->options(function () {
                return \App\Models\News::latest()
                    ->pluck('title', 'id');
            })
            ->searchable()
            ->placeholder('Bir etkinlik haberi seçin (opsiyonel)')
            ->helperText('Bu proje ile ilişkili bir haber varsa seçin.')
            ->columnSpanFull(),


                            Forms\Components\FileUpload::make('image')
                                ->label('Kapak Görseli')
                                ->image()
                                ->directory('projects/covers')
                                ->required()
                                ->columnSpanFull(),
                        Forms\Components\TextInput::make('slug')
                            ->label('SEO URL')
                            ->required()
                            ->unique(ignoreRecord: true),

                        Forms\Components\TextInput::make('external_url')
                            ->label('Dış Yönlendirme (opsiyonel)')
                            ->placeholder('https://...'),
                    ]),

                Forms\Components\RichEditor::make('content')
                    ->label('İçerik Yazısı')
                    ->columnSpanFull(),

                Forms\Components\Repeater::make('form_config')
                    ->label('Başvuru Formu Alanları')
                    ->schema([
                        Forms\Components\TextInput::make('label')->label('Alan Başlığı')->required(),
                        Forms\Components\Select::make('type')
                            ->label('Alan Tipi')
                            ->options([
                                'text' => 'Metin Kutusu',
                                'email' => 'E-posta',
                                'textarea' => 'Açıklama',
                                'file' => 'Dosya',
                            ])
                            ->required(),
                        Forms\Components\Toggle::make('required')
                            ->label('Zorunlu mu?')
                            ->default(false),
                    ])
                    ->columnSpanFull()
                    ->helperText('Başvuru formundaki alanları kullanıcı tanımlar.'),
            ]);
    }


    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->label('Başlık')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('slug')
                    ->label('URL')
                    ->copyable()
                    ->toggleable(),

                Tables\Columns\TextColumn::make('external_url')
                    ->label('Dış Link')
                    ->url(fn ($record) => $record->external_url)
                    ->openUrlInNewTab()
                    ->toggleable(isToggledHiddenByDefault: true),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Oluşturulma')
                    ->dateTime('d.m.Y H:i')
                    ->sortable()
                    ->since(),
            ])
            ->filters([
                // şimdilik boş, istersen yayın durumu vs ekleriz
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Action::make('viewApplications')
        ->label('Başvuruları Gör')
        ->icon('heroicon-o-eye')
        ->modalHeading(fn ($record) => $record->title . ' - Başvurular')
        ->modalSubmitAction(false)
        ->modalCancelActionLabel('Kapat')
        ->modalContent(fn ($record) => view('filament.modals.project-applications-table', [
            'project' => $record,
            'applications' => $record->applications()->latest()->get(),
        ])),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }


    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProjects::route('/'),
            'create' => Pages\CreateProject::route('/create'),
            'edit' => Pages\EditProject::route('/{record}/edit'),
        ];
    }
}
