export interface WithEditorProps {
    editor: any;
}
export interface BaseAssetProvider {
    /**
     * Asset Provider ID.
     * @example "my-provider"
     */
    id: string;
    /**
     * Asset types supported by this provider.
     * Only providers that support the current asset type show up in the asset provider filter.
     * @examples
     * types: 'image',
     * // Or an array of types
     * types: ['image', 'video']
     */
    types: string;
    /**
     * Label to display in the asset provider filter.
     * You may use a function instead to translate this string.
     *
     * @examples
     * label: 'My asset provider'
     * // As a function, for dynamic labels
     * label: ({ editor }) => editor.I18n.t('myProviderLabel')
     */
    label: string | ((props: WithEditorProps) => string);
    /**
     * Search configuration.
     * @examples
     * search: {
     *  // Set this to true if you want AssetProvider.onLoad to retrigger when the user types in the search field. When false, loaded assets are filtered locally by name.
     *  reloadOnInput: true,
     *  // Search value  debounce time in milliseconds
     *  debounceMs: 1000
     * }
     */
    search?: {
        /**
         * Set this to true if you want AssetProvider.onLoad to retrigger when the user types in the search field.
         * When false, loaded assets are filtered locally by name.
         * @defaul false
         */
        reloadOnInput?: boolean;
        /**
         * @default 1000
         */
        debounceMs?: number;
    };
    /**
     * Define how to fetch these assets.
     * Return an array of assets.
     * You may return an array of Page objects to enable endless scrolling, you can rely on the pageIndex argument for this.
     *
     * @examples
     * async () => {
     *   // Simple asset array
     *   return [
     *     { src: 'https://www.example.com/items/1' },
     *     { src: 'https://www.example.com/items/2' },
     *     { src: 'https://www.example.com/items/3' }
     *   ]
     * }
     *
     * async ({ pageIndex }) => {
     *   // Offset based pagination
     *   const pageSize = 20;
     *   const params = new URLSearchParams({ page: pageIndex, pageSize })
     *   const response = await fetch(`https://www.example.com/items?${params}`)
     *   const page = await response.json()
     *   const itemCount = pageSize * pageIndex + page.items.length
     *   return {
     *     items: page.items,
     *     isLastPage: itemCount >= page.total
     *   }
     * }
     *
     * async ({ pageCustomData }) => {
     *   // Token based pagination.
     *   const params = new URLSearchParams({ pageToken: pageCustomData?.token })
     *   const response = await fetch(`https://www.example.com/items?${params}`)
     *   const page = await response.json()
     *   return {
     *     items: page.items,
     *     nextPageCustomData: { token: page.nextPageToken },
     *     isLastPage: !page.nextPageToken
     *   }
     * }
     */
    onLoad: '__fn__';
    /**
     * Custom layout for rendering an asset item in the AssetManager.
     * @examples
     * itemLayout: ({ assetProps, onSelect }) => ({
     *  type: 'column',
     *  style: { height: 150 },
     *  onClick: () => onSelect(assetProps),
     *  children: [
     *    { type: 'custom', render: () => `<img src="${assetProps.src}" style="width: 100%; height: 100%; object-fit: cover">` },
     *    { type: 'text', style: { width: '100%' }, content: assetProps.name ?? '' }
     *  ]
     * })
     */
    itemLayout?: '__fn__';
}
