import type { Page } from 'grapesjs';
import { CommandItem, CustomItems, WithEditorProps } from './common';
export interface PagesConfig {
    /**
     * Custom logic for new page creation.
     * Pass `false` to disable adding new pages.
     * @example
     * add: ({ editor, rename }) => {
     *  // Add new page via GrapesJS API
     *  const page = editor.Pages.add({
     *    name: 'New page',
     *    component: `<div>New page</div>`
     *  }, { select: true });
     *
     *  rename(page);// Trigger rename action
     * },
     * // Disable adding new pages
     * add: false,
     */
    add?: false | ((props: WithEditorProps & {
        rename: (page: Page) => void;
    }) => void);
    /**
     * Custom logic for page duplicate.
     * Pass `false` to disable duplicating pages.
     * @example
     * duplicate: ({ editor, page, rename }) => {
     *  const root = page.getMainComponent();
     *  const clonedPage = editor.Pages.add({
     *    name: `${page.getName()} (Copy)`,
     *    component: root.clone(),
     *  }, { select: true });
     *
     *  rename(clonedPage);// Trigger rename action
     * },
     * // Disable duplicating pages
     * duplicate: false,
     */
    duplicate?: false | ((props: WithEditorProps & {
        rename: (page: Page) => void;
        page: Page;
    }) => void);
    /**
     * Custom logic for page remove.
     * Pass `false` to disable removing pages.
     * @example
     * remove: ({ editor, page }) => {
     *  editor.Pages.remove(page);
     *  alert('Page removed!');
     * },
     * // Disable duplicating pages
     * remove: false,
     */
    remove?: false | ((props: WithEditorProps & {
        page: Page;
    }) => void);
    /**
     * Provide custom command items to pages.
     * @example
     * commandItems: ({ items }) => {
     *  return [
     *    ...items,
     *    {
     *      id: 'test',
     *      label: 'Test',
     *      cmd: ({ page }) => alert(`Custom command ${page.getName()}`)
     *    }
     *  ];
     * }
     * // Disable command items
     * commandItems: false;
     */
    commandItems?: CustomItems<CommandItemPage>;
    /**
     * Allow to import HTML documents (eg. HTML with `doctype`, `head`, etc.).
     */
    allowDocuments?: boolean;
    /**
     * Indicate if the page settings should be shown.
     */
    settings?: boolean;
}
export interface CommandItemPage extends CommandItem<{
    page: Page;
}> {
}
