import type { EditorProps } from '@grapesjs/react';
import type { PluginToLoad } from '@grapesjs/react/dist/utils/plugins';
import type { BlockProperties, Editor, EditorConfig, Page } from 'grapesjs';
import type { ReactNode } from 'react';
import { AssetsConfig } from './typeConfigs/assets';
import { BlocksConfig } from './typeConfigs/blocks';
import { CommandButtonItem, CommandDefaultOptions, WithEditorProps } from './typeConfigs/common';
import { ComponentsConfig } from './typeConfigs/components';
import { DataSourcesConfig } from './typeConfigs/dataSources';
import { DevicesConfig } from './typeConfigs/devices';
import { FontsConfig } from './typeConfigs/fonts';
import { GlobalStylesConfig } from './typeConfigs/globalStyles';
import { LayoutConfig } from './typeConfigs/layout';
import { PagesConfig } from './typeConfigs/pages';
import { ProjectConfig, ProjectData } from './typeConfigs/project';
import { StorageConfig } from './typeConfigs/storage';
import { TemplatesConfig } from './typeConfigs/templates';
import { CustomThemeConfig, ThemeOptions } from './typeConfigs/themes';
export interface CreateEditorOptions {
    /**
     * The license key used to identify your editor project.
     */
    licenseKey: string;
    /**
     * Assets configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/assets/overview
     */
    assets?: AssetsConfig;
    /**
     * Fonts configuration.
     */
    fonts?: FontsConfig;
    /**
     * Storage configuration for projects.
     * @see https://app.grapesjs.com/docs-sdk/configuration/projects#storage
     */
    storage?: StorageConfig;
    /**
     * Project configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/projects
     */
    project?: ProjectConfig;
    /**
     * Identity configuration.
     */
    identity?: IdentityConfig;
    /**
     * Pages configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/pages
     */
    pages?: PagesConfig | false;
    /**
     * Devices configuration.
     */
    devices?: DevicesConfig;
    /**
     * Blocks configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/blocks
     */
    blocks?: BlocksConfig;
    /**
     * I18n configuration.
     */
    i18n?: I18nConfig;
    /**
     * Indicate the local storage key used to store editor-related data (e.g., theme switcher).
     * If not provided, no data will be stored on the user's device.
     * @example
     * localStorage: 'userStudioData'
     */
    localStorage?: string;
    /**
     * **WARNING**: Not yet usable for public SDK.
     *
     * Plugin manager configuration.
     */
    pluginManager?: PluginManagerConfig | boolean;
    /**
     * Array of GrapesJS plugins.
     * Differently from the GrapesJS `plugins` option, this one allows you also to load plugins
     * asynchronously from a CDN URL.
     * @example
     * plugins: [
     *  {
     *    // The id should be name of the plugin that will be actually loaded
     *    id: 'gjs-blocks-basic',
     *    src: 'https://unpkg.com/grapesjs-blocks-basic',
     *    options: {}
     *  }
     *  // plugin already loaded in the global scope (eg. loaded via CND in HTML)
     *  'grapesjs-plugin-forms',
     *  // plugin as a function
     *  myPlugin,
     * ],
     * // you can also provide a custom function and filter default plugins.
     * // WARNING: filtering "system plugins" (passed as functions) might break the editor.
     * plugins: ({ plugins }) => plugins.filter(...),
     */
    plugins?: EditorPluginsToLoad | ((props: {
        plugins: EditorPluginsToLoad;
    }) => EditorPluginsToLoad);
    /**
     * Root element on which to mount the editor.
     * NOTE: to use only for non React projects.
     */
    root?: string | HTMLElement;
    /**
     * Fit wrapper height.
     * Useful in case the element containing the editor or the editor itself
     * doesn't have a static height (eg. not defined in px).
     * @default true
     */
    autoHeight?: boolean;
    /**
     * CDN URL for the GrapesJS JS file or the instance of GrapesJS.
     * @example 'https://unpkg.com/grapesjs@0.20.3'
     */
    gjsScript?: EditorProps['grapesjs'];
    /**
     * CDN URL for the GrapesJS CSS file.
     * @example 'https://unpkg.com/grapesjs@0.20.3/dist/css/grapes.min.css'
     */
    gjsStyle?: EditorProps['grapesjsCss'];
    /**
     * Callback triggered once the editor instance is created.
     */
    onEditor?: EditorProps['onEditor'];
    /**
     * Callback triggered once the editor is ready (mounted and loaded data from the Storage).
     */
    onReady?: EditorProps['onReady'];
    /**
     * Callback triggered on each update in the editor project.
     * The updated ProjectData (JSON) is passed as a first argument.
     */
    onUpdate?: EditorProps['onUpdate'];
    /**
     * Callback triggered when the editor is destroyed.
     */
    onDestroy?: (editor: Editor) => void;
    /**
     * GrapesJS Options.
     */
    gjsOptions?: EditorConfig;
    /**
     * Default editor theme. Possible options: `dark` | `light` | `auto`.
     * @default 'auto'
     * @see https://app.grapesjs.com/docs-sdk/configuration/themes
     */
    theme?: ThemeOptions;
    /**
     * Custom theme for the editor. Override specific props of the default theme.
     *
     * Use the `theme` option to choose the base theme.
     * @see https://app.grapesjs.com/docs-sdk/configuration/themes#custom-theme-colors
     */
    customTheme?: CustomThemeConfig;
    /**
     * Custom layout.
     * @see https://app.grapesjs.com/docs-sdk/configuration/layout/overview
     */
    layout?: LayoutConfig;
    /**
     * Components configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/components/overview
     */
    components?: ComponentsConfig;
    /**
     * Parser configuration.
     * ## WARNING: work in progress.
     */
    parser?: {
        /**
         * Indicate if the parser errors should be shown in the editor.
         * @default true
         */
        showErrors?: boolean;
    };
    /**
     * Provide custom action buttons.
     * @deprecated Use `layouts` configuration.
     * @example
     * actions: [
     *  {
     *   id: 'my-action',
     *   onClick: ({ editor }) => alert('Current HTML:', editor.getHtml()),
     *   tooltip: 'My action',
     *   icon: '<svg>...</svg>',
     *  }
     * ],
     * // you can also filter default actions.
     * actions: ({ actions }) => {
     *  const filtered = actions.filter(a => a.id !== 'preview');
     *  return [
     *    ...filtered,
     *    {....} // your other actions
     *  ]
     * },
     */
    actions?: CommandButtonItem[] | ((props: {
        actions: CommandButtonItem[];
    } & WithEditorProps) => CommandButtonItem[]) | false;
    commands?: {
        options?: CommandDefaultOptions;
    };
    /**
     * Customize settings menu items (gear icon).
     * @example
     * // Hide "Save Project" and the theme switcher
     * settingsMenu: {
     *  saveProject: false,
     *  theme: false,
     * }
     * // Hide settings menu button
     * settingsMenu: false
     */
    settingsMenu?: SettingsMenuProps | false;
    /**
     * Design panel configuration.
     * @deprecated Use `layouts` configuration.
     */
    designPanel?: DesignPanelConfig;
    /** @private */
    updateAppShell?: boolean;
    /** @private */
    serviceWorker?: boolean;
    /** @private */
    debug?: boolean;
    stg?: boolean;
    /**
     * Global styles configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/global-styles
     */
    globalStyles?: GlobalStylesConfig;
    /**
     * Templates configuration.
     * @see https://app.grapesjs.com/docs-sdk/configuration/templates
     */
    templates?: TemplatesConfig;
    /**
     * Datasources configuration.
     */
    dataSources?: DataSourcesConfig;
}
export interface CreateEditorHeadlessOptions extends Omit<CreateEditorOptions, 'gjsScript' | 'root' | 'plugins'> {
    /**
     * GrapesJS constructor is required for the headless editor.
     * @example
     * import grapesjs from 'grapesjs';
     *
     * // ...
     * createStudioHeadlessEditor({
     *  gjsScript: grapesjs,
     *  // ...
     * })
     */
    gjsScript: Exclude<EditorProps['grapesjs'], string>;
    /**
     * GrapesJS plugins.
     *
     * **WARNING**: Only plugins as function are supported by the headless editor.
     * @example
     * plugins: [
     *  (editor) => {...}
     * ]
     */
    plugins?: EditorHeadlessPlugins | ((props: {
        plugins: EditorHeadlessPlugins;
    }) => EditorHeadlessPlugins);
    /**
     * Project data JSON to load in the headless editor.
     * @example
     * projectData: yourProjectJSON // { pages: [...], }
     */
    projectData?: ProjectData;
}
export type EditorPluginsToLoad = Required<EditorProps>['plugins'];
export type EditorHeadlessPlugins = Exclude<EditorPluginsToLoad[0], PluginToLoad | string>[];
export interface IdentityConfig {
    /**
     * A unique id for the end user of the editor.
     * Mandatory for project/asset cloud.
     * Avoid easy to guess, public identifiers or sensitive data here, like emails.
     */
    id?: string;
}
export interface PluginManagerConfig {
}
export interface PanelPageSettingsState {
    isOpen: boolean;
    width: number;
    page?: Page;
}
export interface I18nConfig {
    /**
     * Custom locales for the editor.
     * @example
     * locales: {
     *  en: { delete: 'Delete EN' }
     * }
     */
    locales?: Record<string, Record<string, any>>;
}
export interface SettingsMenuProps {
    about?: boolean;
    embed?: boolean;
    openProject?: boolean;
    saveProject?: boolean;
    loadProject?: boolean;
    installApp?: boolean;
    theme?: boolean;
}
export interface BlockStudioProperties extends BlockProperties {
    mediaReact?: ReactNode;
    full?: boolean;
}
export interface DesignPanelConfig {
    /**
     * Indicate if the component properties should be shown in the styles panel.
     */
    propsInStyles?: boolean;
    /**
     * Hide tabs from the design panel
     */
    hideTabs?: boolean;
}
export { StudioCommands, StudioEvents } from './plugins/global/types';
export { ToastVariant } from './components/Toast';
export type { IToast as ToastOptions } from './components/Toast';
export type { CustomTraitRenderFn } from './components/TraitManager/TraitCustomField';
export type { DialogOptions } from './store/modalStore';
export * from './typeConfigs/assets';
export * from './typeConfigs/common';
export * from './typeConfigs/components';
export * from './typeConfigs/dataSources';
export * from './typeConfigs/devices';
export * from './typeConfigs/fonts';
export * from './typeConfigs/globalStyles';
export * from './typeConfigs/layout';
export * from './typeConfigs/layoutComponents';
export * from './typeConfigs/pages';
export * from './typeConfigs/project';
export * from './typeConfigs/rte';
export * from './typeConfigs/shared';
export * from './typeConfigs/storage';
export * from './typeConfigs/templates';
export * from './typeConfigs/themes';
